﻿<%@ WebHandler Language="C#" Class="MyCounter" %>

// 2009-7-25    新規作成

using System;
using System.Web;
using System.IO;
using System.Drawing;
using System.Drawing.Imaging;

public class MyCounter : IHttpHandler
{
    private string _counterName;         // カウンターの名前
    private string _dataFileName;        // 設定ファイルの名前

    private Bitmap[] _png_images = new Bitmap[10];      // 0 ～ 9 の数字画像
    private Size _size;                                 // 数字画像ひとつの幅と高さ
    
    public void ProcessRequest (HttpContext context)
    {
        try
        {
            if (!Check(context))
            {
                return;
            }
            Init(context);
            Shori(context);
        }
        catch (Exception ex)
        {
            WriteExceptionInfo(context, ex);
            Error(context, 0);      // 何かよくわからないが、例外が出てる！
        }
        finally
        {
            Term();
        }
    }

    public bool IsReusable
    {
        get
        {
            return true;
        }
    }

    /// <summary>
    /// 事前のチェックを行います。
    /// </summary>
    /// <param name="context"></param>
    /// <returns>チェック合格なら true を返します。</returns>
    private bool Check(HttpContext context)
    {
        _counterName = context.Request.QueryString["n"];
        if (IsValidCounterName(_counterName) == false)
        {
            Error(context, 1);      // カウンターの名前が不正
            return false;
        }

        _dataFileName = GetDataFileName(context, _counterName);
        if (File.Exists(_dataFileName) == false)
        {
            Error(context, 2);      // 設定ファイルが存在しない
            return false;
        }

        return true;
    }
    
    /// <summary>
    /// 準備
    /// </summary>
    /// <param name="context"></param>
    private void Init(HttpContext context)
    {
        // ---------------
        // 数字画像ファイルを読む
        // ---------------
        Read_NumberSymbols(context);
    }

    /// <summary>
    /// 後片付け
    /// </summary>
    private void Term()
    {
        // ---------------
        // 数字画像ファイルを開放する
        // ---------------
        foreach (Bitmap bmp in _png_images)
        {
            if (bmp != null)
            {
                bmp.Dispose();
            }
        }
    }
    
    /// <summary>
    /// 処理します。
    /// </summary>
    /// <param name="context"></param>
    private void Shori(HttpContext context)
    {
        // ---------------
        // 設定ファイルを読み書きする
        // ---------------
        int counterValue;           // カウンターの値。設定ファイルの一行目に記載。
        int counterWidth;           // カウンターの桁数。設定ファイルの二行目に記載。
        Increment(_dataFileName, out counterValue, out counterWidth);

        // ---------------
        // 
        // ---------------
        string fmt_str = "";        // フォーマット用文字列
        for (int i = 0; i < counterWidth; i++)
        {
            fmt_str += "0";
        }

        // ---------------
        // 
        // ---------------
        string counterString;       // カウンター文字列。例えば 00045 のように先頭が 0 埋め。
        counterString = counterValue.ToString(fmt_str);

        // ---------------
        // カウンター画像を作ってクライアントに返す
        // ---------------
        var newBitmap = new Bitmap(
		    _size.Width * counterWidth,
			_size.Height
			);
        using (newBitmap)
        {
            var g = Graphics.FromImage(newBitmap);
            using (g)
            {
                for (int i = 0; i < counterString.Length; i++)
                {
                    char c = counterString[i];
                    int num = int.Parse(c + "");
                    g.DrawImage(_png_images[num], _size.Width * i, 0);
                }
            }

            GeneratePngImage(context, newBitmap);
        }

    }

    /// <summary>
    /// 数字の画像ファイル (0 ～ 9) を読み込みます。
    /// </summary>
    /// <param name="context"></param>
    private void Read_NumberSymbols(HttpContext context)
    {
        string path = context.Server.MapPath("~/image");
        
        for (int i = 0; i < 10; i++)
        {
            _png_images[i] = new Bitmap(path + "\\" + i + ".png");
        }
        
        _size.Width = _png_images[0].Width;
        _size.Height = _png_images[0].Height;
    }

    /// <summary>
    /// カウンターの名前が適切かどうかチェックします。
    /// </summary>
    /// <param name="counterName">カウンターの名前</param>
    /// <returns>カウンターの名前が適切なら true を返します。</returns>
    private bool IsValidCounterName(string counterName)
    {
        if (string.IsNullOrEmpty(counterName))
        {
            return false;
        }
        
        const string CHARS = "abcdefghijklmnopqrstuvwxyz0123456789";

        foreach (char c in counterName)
        {
            if (CHARS.IndexOf(c) == -1)
            {
                return false;
            }
        }
        
        return true;
    }

    /// <summary>
    /// 設定ファイルの完全パスを得ます。
    /// </summary>
    /// <param name="context"></param>
    /// <param name="counterName">カウンターの名前</param>
    /// <returns>設定ファイルの完全パス</returns>
    private string GetDataFileName(HttpContext context, string counterName)
    {
        return context.Server.MapPath("~/App_Data/" + counterName + ".txt");
    }

    /// <summary>
    /// エラー番号を描いた画像を作ります。
    /// </summary>
    /// <param name="context"></param>
    /// <param name="errorID">エラー番号</param>
    private void Error(HttpContext context, int errorID)
    {
        var bitmap = new Bitmap(100, 20);
        using (bitmap)
        {
            var g = Graphics.FromImage(bitmap);
            using (g)
            {
                var font = new Font("MS UI Gothic", 9);
                using (font)
                {
                    g.DrawString("ERROR " + errorID, font, Brushes.Black, 0, 0);
                }
            }

            GeneratePngImage(context, bitmap);
        }
    }

    /// <summary>
    /// 画像を作ります。
    /// </summary>
    /// <param name="context"></param>
    /// <param name="bitmap"></param>
    private static void GeneratePngImage(HttpContext context, Bitmap bitmap)
    {
        
#if true
        var memStream = new MemoryStream();
        using (memStream)
        {
            bitmap.Save(memStream, ImageFormat.Png);

            context.Response.Cache.SetCacheability(HttpCacheability.NoCache);
            context.Response.ContentType = "image/png";
            
            //context.Response.BinaryWrite(memStream.ToArray());    // どちらでも OK
            memStream.WriteTo(context.Response.OutputStream);       // どちらでも OK
            
            context.Response.Flush();
        }
#else
        context.Response.Cache.SetCacheability(HttpCacheability.NoCache);
        context.Response.ContentType = "image/png";
        bitmap.Save(context.Response.OutputStream, ImageFormat.Png);
        context.Response.Flush();
#endif
        
        // ここでわざわざ MemoryStream を使う理由は以下を参照のこと。
        //
        // Dynamic PNG images with ASP.NET and GDI+
        // http://aspalliance.com/319
        // 
        // Q: Why do I get "A generic error occurred in GDI+. " when 
        //    I switch my working GIF to use PNG format instead?
        // A: You can't use the Bitmap Save() method with a "non-seekable" stream. 
        //    Some image formats require that the stream can seek. 

    }

    /// <summary>
    /// 例外をテキスト ファイルに記録します。
    /// </summary>
    /// <param name="context"></param>
    /// <param name="ex">例外</param>
    private void WriteExceptionInfo(HttpContext context, Exception ex)
    {
        var dt = DateTime.Now.ToString("yyyyMMdd.HHmmss.fff");
        string filename = context.Server.MapPath("~/App_Data/" + dt + ".txt");
        using (var s = new StreamWriter(filename))
        {
            s.WriteLine(dt);
            s.WriteLine(ex.ToString());
        }
    }
    
    /// <summary>
    /// 設定ファイルからカウンターの値と幅を読み、カウンターの値をインクリメントします。
    /// </summary>
    /// <param name="dataFileName">設定ファイルの完全パス</param>
    /// <param name="counterValue">[out] インクリメント後のカウンター値</param>
    /// <param name="width">[out] カウンターの幅</param>
    private void Increment(string dataFileName, out int counterValue, out int width)
    {
        FileStream fs = null;

        for (int i = 0; i < 10; i++)
        {
            try
            {
                fs = new FileStream(
					dataFileName,
					FileMode.Open,              // ファイルを開きます。ファイルが無いなら死亡。
                    FileAccess.ReadWrite,       // ファイルを読み書きします。
                    FileShare.None              // 共有を拒否します。
                    );
                
                break;
            }
            catch (IOException)
            {
                System.Threading.Thread.Sleep(new Random().Next(1000));
            }
        }
        
        using (fs)
        {
            // -----------------
            // 読む
            // -----------------
            var reader = new StreamReader(fs);
            string line1 = reader.ReadLine();
            string line2 = reader.ReadLine();
            if (int.TryParse(line1, out counterValue) == false)
            {
                counterValue = 0;       // デフォルトのカウンター値は 0
            }
            if (int.TryParse(line2, out width) == false)
            {
                width = 5;              // デフォルトのカウンター桁数は 5
            }

            // -----------------
            // インクリメント
            // -----------------
            counterValue++;

            // -----------------
            // 書く
            // -----------------
            fs.SetLength(0);
            var writer = new StreamWriter(fs);
            writer.WriteLine(counterValue);
            writer.WriteLine(width);
            writer.Flush();
        }

    }
}
